<?php
session_start();
include __DIR__ . "/db.php";

if(!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'student'){
    header("Location: login.php");
    exit;
}

$student_id = $_SESSION['user_id'];
$course = $_SESSION['course'];
$msg = "";

// Handle assignment submission
if($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['lesson_id'])){
    $lesson_id = intval($_POST['lesson_id']);

    // Check lesson exists
    $stmt_check = $conn->prepare("SELECT id FROM lessons WHERE id=? AND course=?");
    $stmt_check->bind_param("is", $lesson_id, $course);
    $stmt_check->execute();
    $res_check = $stmt_check->get_result();
    if($res_check->num_rows === 0){
        $msg = "❌ Lesson does not exist or not in your course.";
    } else {
        // Handle file upload
        if(isset($_FILES['assignment_file']) && $_FILES['assignment_file']['error'] === UPLOAD_ERR_OK){
            $upload_dir = __DIR__ . "/assignments/";
            if(!is_dir($upload_dir)) mkdir($upload_dir, 0755, true);

            $file_tmp = $_FILES['assignment_file']['tmp_name'];
            $file_name = uniqid("assignment_")."_".basename($_FILES['assignment_file']['name']);
            $file_path = "assignments/".$file_name;
            move_uploaded_file($file_tmp, $upload_dir.$file_name);

            $stmt = $conn->prepare("INSERT INTO assignments(student_id, lesson_id, course, file_path, submitted_at) VALUES(?,?,?,?,NOW())");
            $stmt->bind_param("iiss", $student_id, $lesson_id, $course, $file_path);
            if($stmt->execute()){
                $msg = "✅ Assignment submitted successfully!";
            } else {
                $msg = "❌ Database error: ".$conn->error;
            }
        } else {
            $msg = "⚠️ Please select a file to upload.";
        }
    }
}

// Fetch lessons
$stmt = $conn->prepare("SELECT l.id, l.title, l.youtube_link, l.content, l.created_at, u.fullname AS admin_name
                        FROM lessons l
                        JOIN users u ON l.admin_id = u.id
                        WHERE l.course=?
                        ORDER BY l.created_at DESC");
$stmt->bind_param("s", $course);
$stmt->execute();
$lessons = $stmt->get_result();
?>

<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>Lessons - Xploraa Academy</title>
<style>
body{font-family:Poppins,sans-serif;margin:0;background:#f5f7fa;color:#222;}
.navbar{display:flex;justify-content:space-between;background:#003366;color:#fff;padding:15px 40px;}
.navbar a{color:white;text-decoration:none;padding:8px 16px;}
.navbar a:hover{background:#ff6600;border-radius:8px;}
.container{max-width:900px;margin:50px auto;padding:0 20px;}
.card{background:white;padding:20px;border-radius:12px;box-shadow:0 0 12px rgba(0,0,0,0.1);margin-bottom:20px;animation:fadeIn 0.5s ease-in;}
input,button{width:100%;margin:10px 0;padding:10px;border-radius:8px;border:1px solid #ccc;}
button{background:#003366;color:white;border:none;cursor:pointer;}
button:hover{background:#ff6600;}
.success{background:#d4edda;color:#155724;padding:10px;border-radius:8px;font-weight:bold;}
.error{background:#f8d7da;color:#721c24;padding:10px;border-radius:8px;font-weight:bold;}
@keyframes fadeIn{from{opacity:0;transform:translateY(20px);}to{opacity:1;transform:translateY(0);}}
</style>
</head>
<body>

<div class="navbar">
<div class="logo" style="color:#ff6600;font-weight:bold;">Xploraa Academy</div>
<div>
<a href="dashboard.php">Dashboard</a>
<a href="lessons.php">Lessons</a>
<a href="profile.php">Profile</a>
<a href="student_messages.php">Messages</a>
<a href="logout.php">Logout</a>
</div>
</div>

<div class="container">
<h2>Lessons for <?= htmlspecialchars($course) ?></h2>

<?php if($msg): ?>
<div class="<?= strpos($msg,'❌')!==false || strpos($msg,'⚠️')!==false ? 'error' : 'success' ?>">
<?= htmlspecialchars($msg) ?>
</div>
<?php endif; ?>

<?php if($lessons->num_rows > 0): ?>
    <?php while($l = $lessons->fetch_assoc()): ?>
    <div class="card">
        <h3><?= htmlspecialchars($l['title']) ?></h3>
        <p><strong>Posted by:</strong> <?= htmlspecialchars($l['admin_name']) ?> | <small><?= $l['created_at'] ?></small></p>
        <?php if($l['youtube_link']): ?>
            <p><a href="<?= htmlspecialchars($l['youtube_link']) ?>" target="_blank">📺 Watch Video</a></p>
        <?php endif; ?>
        <p><?= nl2br(htmlspecialchars($l['content'])) ?></p>

        <form method="POST" enctype="multipart/form-data">
            <input type="hidden" name="lesson_id" value="<?= $l['id'] ?>">
            <input type="file" name="assignment_file" required>
            <button type="submit">Submit Assignment</button>
        </form>
    </div>
    <?php endwhile; ?>
<?php else: ?>
<p>No lessons posted yet.</p>
<?php endif; ?>

</div>

<footer style="background:#003366;color:white;text-align:center;padding:25px 0;margin-top:50px;">
&copy; <?= date('Y'); ?> <span style="color:#ff6600;font-weight:bold;">Xploraa Academy</span>
</footer>

</body>
</html>
