<?php
session_start();
include __DIR__ . "/db.php";

if(!isset($_SESSION['user_id']) || ($_SESSION['role'] ?? '') !== 'student'){
    header("Location: login.php");
    exit;
}

$student_id = (int)$_SESSION['user_id'];
$course     = $_SESSION['course'] ?? '';
$msg = "";

/** Helper: escape */
function h($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }

/** Helper: make YouTube embed src from many URL styles */
function youtube_embed_src(?string $url): ?string {
    if (!$url) return null;
    $url = trim($url);

    // Already-embed
    if (stripos($url, 'youtube.com/embed/') !== false) {
        return $url;
    }

    $parts = @parse_url($url);
    if (!$parts || empty($parts['host'])) return null;

    $host  = strtolower($parts['host']);
    $path  = $parts['path']  ?? '';
    $query = $parts['query'] ?? '';

    $id = '';

    // youtu.be/<id>
    if (strpos($host, 'youtu.be') !== false) {
        $id = ltrim($path, '/');
    }
    // youtube.com/*
    elseif (strpos($host, 'youtube.com') !== false || strpos($host, 'www.youtube.com') !== false) {
        if (strpos($path, '/watch') === 0) {
            parse_str($query, $qs);
            $id = $qs['v'] ?? '';
        } elseif (strpos($path, '/shorts/') === 0) {
            $id = basename($path);
        } elseif (strpos($path, '/embed/') === 0) {
            $id = basename($path);
        }
    }

    $id = preg_replace('/[^A-Za-z0-9_-]/', '', $id ?? '');
    if ($id === '') return null;

    return 'https://www.youtube.com/embed/' . $id;
}

// Handle assignment submission (file upload)
if($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['lesson_id'])){
    $lesson_id = intval($_POST['lesson_id']);

    // Check lesson exists and belongs to course
    $stmt_check = $conn->prepare("SELECT id FROM lessons WHERE id=? AND course=?");
    $stmt_check->bind_param("is", $lesson_id, $course);
    $stmt_check->execute();
    $res_check = $stmt_check->get_result();
    if($res_check->num_rows === 0){
        $msg = "❌ Lesson does not exist or not in your course.";
    } else {
        // Handle file upload
        if(isset($_FILES['assignment_file']) && $_FILES['assignment_file']['error'] === UPLOAD_ERR_OK){
            $upload_dir = __DIR__ . "/assignments/";
            if(!is_dir($upload_dir)) mkdir($upload_dir, 0755, true);

            $file_tmp  = $_FILES['assignment_file']['tmp_name'];
            $safe_name = preg_replace('/[^A-Za-z0-9._-]/','_', basename($_FILES['assignment_file']['name']));
            $file_name = uniqid("assignment_")."_".$safe_name;

            $file_path_rel = "assignments/".$file_name;
            $file_path_abs = $upload_dir.$file_name;

            if (move_uploaded_file($file_tmp, $file_path_abs)) {
                $stmt = $conn->prepare("INSERT INTO assignments(student_id, lesson_id, course, file_path, submitted_at) VALUES(?,?,?,?,NOW())");
                $stmt->bind_param("iiss", $student_id, $lesson_id, $course, $file_path_rel);
                if($stmt->execute()){
                    $msg = "✅ Assignment submitted successfully!";
                } else {
                    $msg = "❌ Database error: ".$conn->error;
                }
            } else {
                $msg = "⚠️ Upload failed. Please try again.";
            }
        } else {
            $msg = "⚠️ Please select a file to upload.";
        }
    }
}

// Fetch lessons
$stmt = $conn->prepare("SELECT l.id, l.title, l.youtube_link, l.content, l.created_at, u.fullname AS admin_name
                        FROM lessons l
                        JOIN users u ON l.admin_id = u.id
                        WHERE l.course=?
                        ORDER BY l.created_at DESC");
$stmt->bind_param("s", $course);
$stmt->execute();
$lessons = $stmt->get_result();

// Collect lesson IDs & store lesson rows
$lesson_ids = [];
$all_lessons = [];
while ($row = $lessons->fetch_assoc()) {
    $lesson_ids[] = (int)$row['id'];
    $all_lessons[] = $row;
}

/* Fetch this student's submissions for these lessons (grouped) */
$submissions_by_lesson = [];
if (!empty($lesson_ids)) {
    $placeholders = implode(',', array_fill(0, count($lesson_ids), '?'));
    $types = 'is' . str_repeat('i', count($lesson_ids)); // student_id(int), course(string), lesson_ids(int...)
    $sql = "SELECT id, student_id, lesson_id, course, file_path, submitted_at
            FROM assignments
            WHERE student_id=? AND course=? AND lesson_id IN ($placeholders)
            ORDER BY submitted_at DESC";
    $stmtA = $conn->prepare($sql);
    if ($stmtA) {
        $params = [];
        $params[] = &$types;
        $params[] = &$student_id;
        $params[] = &$course;
        foreach ($lesson_ids as $k => $lid) {
            $params[] = &$lesson_ids[$k];
        }
        call_user_func_array([$stmtA, 'bind_param'], $params);
        $stmtA->execute();
        $resA = $stmtA->get_result();
        while ($a = $resA->fetch_assoc()) {
            $lid = (int)$a['lesson_id'];
            if (!isset($submissions_by_lesson[$lid])) $submissions_by_lesson[$lid] = [];
            $submissions_by_lesson[$lid][] = $a;
        }
    }
}

/* Fetch admin-posted assignment questions for these lessons (grouped) */
$questions_by_lesson = [];
if (!empty($lesson_ids)) {
    $placeholders = implode(',', array_fill(0, count($lesson_ids), '?'));
    $types = str_repeat('i', count($lesson_ids));
    $sqlQ = "SELECT id, lesson_id, question, option_a, option_b, option_c, option_d, marks
             FROM questions
             WHERE lesson_id IN ($placeholders)
             ORDER BY id ASC";
    $stmtQ = $conn->prepare($sqlQ);
    if ($stmtQ) {
        // dynamic bind
        $params = [];
        $params[] = &$types;
        foreach ($lesson_ids as $k => $lid) {
            $params[] = &$lesson_ids[$k];
        }
        call_user_func_array([$stmtQ, 'bind_param'], $params);
        $stmtQ->execute();
        $resQ = $stmtQ->get_result();
        while ($q = $resQ->fetch_assoc()) {
            $lid = (int)$q['lesson_id'];
            if (!isset($questions_by_lesson[$lid])) $questions_by_lesson[$lid] = [];
            $questions_by_lesson[$lid][] = $q;
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>Lessons - Xploraa Academy</title>
<style>
  body{font-family:Poppins,system-ui,-apple-system,Segoe UI,Roboto,Arial,sans-serif;margin:0;background:#f5f7fa;color:#222;}
  .navbar{display:flex;justify-content:space-between;align-items:center;background:#003366;color:#fff;padding:15px 40px;gap:12px;flex-wrap:wrap}
  .navbar a{color:white;text-decoration:none;padding:8px 16px;border-radius:8px;display:inline-block}
  .navbar a:hover{background:#ff6600;}
  .container{max-width:900px;margin:50px auto;padding:0 20px;}
  .card{background:white;padding:20px;border-radius:12px;box-shadow:0 0 12px rgba(0,0,0,0.1);margin-bottom:20px;animation:fadeIn .5s ease-in;}
  input,button{width:100%;margin:10px 0;padding:10px;border-radius:8px;border:1px solid #ccc;}
  button{background:#003366;color:white;border:none;cursor:pointer;font-weight:600}
  button:hover{background:#ff6600;}
  .success{background:#d4edda;color:#155724;padding:10px;border-radius:8px;font-weight:bold;}
  .error{background:#f8d7da;color:#721c24;padding:10px;border-radius:8px;font-weight:bold;}
  @keyframes fadeIn{from{opacity:0;transform:translateY(20px);}to{opacity:1;transform:translateY(0);}}

  /* Responsive YouTube embed */
  .yt-embed{position:relative;width:100%;padding-bottom:56.25%; /* 16:9 */ background:#000;border-radius:12px;overflow:hidden;margin:12px 0 4px 0;box-shadow:0 6px 18px rgba(0,0,0,.15)}
  .yt-embed iframe{position:absolute;inset:0;width:100%;height:100%;border:0;}
  .posted-meta{color:#555;font-size:.9rem;margin-top:-4px}

  .submissions{background:#f9fafc;border:1px solid #e8ecf3;border-radius:10px;padding:12px;margin-top:8px}
  .submissions h4{margin:0 0 10px 0;font-size:1rem;color:#003366}
  .submissions ul{margin:0;padding-left:18px}
  .submissions li{margin:6px 0}
  .pill{display:inline-block;font-size:.78rem;background:#eaf4ff;color:#003366;border:1px solid #cfe3ff;padding:2px 8px;border-radius:999px}

  .questions{background:#fffdf7;border:1px solid #f3e9c8;border-radius:10px;padding:12px;margin-top:12px}
  .questions h4{margin:0 0 10px 0;font-size:1rem;color:#8a5a00}
  .q-item{margin:10px 0;padding:10px;border-radius:8px;background:#fff8e6;border:1px solid #f2dfb4}
  .q-stem{font-weight:600;margin-bottom:6px}
  .q-opts{margin:0;padding-left:18px}
  .q-opts li{margin:3px 0}
  .q-marks{font-size:.8rem;color:#6b4d00}
</style>
</head>
<body>

<div class="navbar">
  <div class="logo" style="color:#ff6600;font-weight:bold;">Xploraa Academy ✨</div>
  <div>
    <a href="dashboard.php">Dashboard</a>
    <a href="lessons.php">Lessons</a>
    <a href="profile.php">Profile</a>
    <a href="student_messages.php">Messages</a>
    <a href="logout.php">Logout</a>
  </div>
</div>

<div class="container">
  <h2>Lessons for <?= h($course) ?> 📚</h2>

  <?php if($msg): ?>
    <div class="<?= (strpos($msg,'❌')!==false || strpos($msg,'⚠️')!==false) ? 'error' : 'success' ?>">
      <?= h($msg) ?>
    </div>
  <?php endif; ?>

  <?php if(!empty($all_lessons)): ?>
    <?php foreach($all_lessons as $l): ?>
      <div class="card">
        <h3><?= h($l['title']) ?></h3>
        <p class="posted-meta"><strong>Posted by:</strong> <?= h($l['admin_name']) ?> • <small><?= h($l['created_at']) ?></small></p>

        <?php if($embed = youtube_embed_src($l['youtube_link'] ?? '')): ?>
          <div class="yt-embed">
            <iframe
              src="<?= h($embed) ?>"
              title="YouTube video player"
              allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture; web-share"
              allowfullscreen></iframe>
          </div>
        <?php elseif(!empty($l['youtube_link'])): ?>
          <p>📺 <a href="<?= h($l['youtube_link']) ?>" target="_blank" rel="noopener">Watch on YouTube</a></p>
        <?php endif; ?>

        <?php if(!empty($l['content'])): ?>
          <p><?= nl2br(h($l['content'])) ?></p>
        <?php endif; ?>

        <!-- Assignment Questions (from admin) -->
        <div class="questions">
          <h4>📘 Assignment Questions</h4>
          <?php
            $lid = (int)$l['id'];
            $qlist = $questions_by_lesson[$lid] ?? [];
          ?>
          <?php if(empty($qlist)): ?>
            <span class="pill">No questions posted yet</span>
          <?php else: ?>
            <?php foreach($qlist as $i => $q): ?>
              <div class="q-item">
                <div class="q-stem">Q<?= $i+1; ?>. <?= nl2br(h($q['question'])) ?></div>
                <ul class="q-opts">
                  <li>A) <?= h($q['option_a']) ?></li>
                  <li>B) <?= h($q['option_b']) ?></li>
                  <li>C) <?= h($q['option_c']) ?></li>
                  <li>D) <?= h($q['option_d']) ?></li>
                </ul>
                <div class="q-marks">Marks: <?= (int)$q['marks'] ?></div>
              </div>
            <?php endforeach; ?>
          <?php endif; ?>
        </div>

        <!-- File upload -->
        <form method="POST" enctype="multipart/form-data">
          <input type="hidden" name="lesson_id" value="<?= (int)$l['id'] ?>">
          <input type="file" name="assignment_file" required>
          <button type="submit">Submit Assignment ✅</button>
        </form>

        <!-- Your submissions -->
        <div class="submissions">
          <h4>📝 Your submissions</h4>
          <?php
            $subs = $submissions_by_lesson[$lid] ?? [];
          ?>
          <?php if(empty($subs)): ?>
            <span class="pill">No submissions yet</span>
          <?php else: ?>
            <ul>
              <?php foreach($subs as $s):
                $fileUrl = $s['file_path'];
                $fileName = basename($fileUrl);
              ?>
                <li>
                  <a href="<?= h($fileUrl) ?>" target="_blank" rel="noopener"><?= h($fileName) ?></a>
                  <small> • submitted <?= h($s['submitted_at']) ?></small>
                </li>
              <?php endforeach; ?>
            </ul>
          <?php endif; ?>
        </div>

      </div>
    <?php endforeach; ?>
  <?php else: ?>
    <p>No lessons posted yet.</p>
  <?php endif; ?>
</div>

<footer style="background:#003366;color:white;text-align:center;padding:25px 0;margin-top:50px;">
  &copy; <?= date('Y'); ?> <span style="color:#ff6600;font-weight:bold;">Xploraa Academy</span> • Keep learning! 🚀
</footer>

</body>
</html>
