<?php
// hookes_law.php — Hooke’s Law / Elasticity 🎮🧷
// Verify Hooke’s law (F = kx): add masses, measure extension, estimate k.

session_start();
if (!isset($_SESSION['k_true'])) {
  $_SESSION['k_true'] = mt_rand(60, 200) / 10.0; // 6.0–20.0 N/m
}
$k_true = (float)$_SESSION['k_true'];
?>
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <title>Hooke’s Law — Virtual Spring Lab 🎮🧷</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <!-- Bootstrap 5 -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <style>
    body{ background:#eef7ff; }
    .lab-wrap{ background:#fff; border-radius:1rem; box-shadow:0 6px 24px rgba(0,0,0,.06); overflow:hidden; }
    .lab-top{ background:linear-gradient(135deg,#0d6efd 0%, #6ea8fe 100%); color:#fff; }
    .chip{ font-size:.8rem; background:#e7f1ff; color:#084298; border-radius:999px; padding:.25rem .6rem; }

    .scene{ position:relative; height:360px; background:#f8fbff; border-left:8px solid #d0e3ff; border-right:8px solid #d0e3ff; border-bottom:8px solid #d0e3ff; }
    .ceiling{ height:18px; background:#0d6efd; }
    .spring{ position:absolute; left:50%; transform:translateX(-50%); width:24px;
      background:repeating-linear-gradient(to bottom, #1f6feb 0 6px, #cfe2ff 6px 12px); border-radius:12px; transition:height .25s ease; }
    .mass{ position:absolute; left:50%; transform:translateX(-50%); width:60px; height:60px; background:#2b3035; border-radius:8px; box-shadow: inset 0 2px 0 rgba(255,255,255,.15); color:#fff; display:flex; align-items:center; justify-content:center; font-weight:700; }
    .ruler{ position:absolute; left:8px; top:30px; bottom:8px; width:36px; background:linear-gradient(#fefefe,#f5f7fb); border-right:2px solid #cbd5e1; border-radius:6px; font-size:.7rem; color:#0f172a; padding-top:6px; }
    .ruler .tick{ position:absolute; left:0; width:100%; border-top:1px solid #0f172a22; }
    .ruler .label{ position:absolute; right:4px; transform:translateY(-50%); }
    table th, table td { white-space:nowrap; }
    .scroll-x{ overflow-x:auto; }
  </style>
</head>
<body>
  <div class="container py-4">
    <a href="index.php" class="btn btn-link mb-3">⬅️ Back to index</a>

    <div class="lab-wrap">
      <div class="lab-top p-3">
        <div class="d-flex flex-wrap align-items-center justify-content-between gap-2">
          <div>
            <h1 class="h4 mb-0">Hooke’s Law — Virtual Spring Lab 🎮🧷</h1>
            <small>F = kx · Measure extension vs force to find k (spring constant)</small>
          </div>
          <span class="chip">SS1 • Elasticity ✅</span>
        </div>
      </div>

      <div class="p-3 p-lg-4">
        <div class="row g-4">
          <!-- Scene -->
          <div class="col-12 col-lg-6">
            <div class="scene rounded">
              <div class="ceiling"></div>
              <div class="ruler" id="ruler"></div>
              <div id="spring" class="spring" style="top:18px; height:120px;"></div>
              <div id="mass" class="mass" style="top:138px;">0 g</div>
            </div>
            <div class="text-center text-muted mt-2">
              <small>Visual only — true scale is shown numerically below.</small>
            </div>
          </div>

          <!-- Controls -->
          <div class="col-12 col-lg-6">
            <div class="card shadow-sm">
              <div class="card-body">
                <div class="row g-2">
                  <div class="col-6">
                    <label class="form-label mb-1">Mass (g) 🧊</label>
                    <div class="input-group">
                      <button class="btn btn-outline-secondary" id="btnDec">−50g</button>
                      <input type="number" class="form-control" id="massInput" value="0" min="0" step="50">
                      <button class="btn btn-outline-secondary" id="btnInc">+50g</button>
                    </div>
                    <div class="form-text">Add/remove 50 g blocks (0–1000 g).</div>
                  </div>
                  <div class="col-6">
                    <label class="form-label mb-1">Gravity (m/s²) 🌍</label>
                    <input type="number" class="form-control" id="gInput" value="9.81" step="0.01" min="9.0" max="9.9">
                    <div class="form-text">Keep 9.81 for standard lab work.</div>
                  </div>
                  <div class="col-12 d-grid gap-2 mt-2">
                    <button class="btn btn-primary" id="btnMeasure">📸 Measure (add reading)</button>
                    <button class="btn btn-outline-primary" id="btnCompute">📐 Compute k from readings</button>
                    <button class="btn btn-success" id="btnCheck">✅ Check my k vs true value</button>
                    <button class="btn btn-danger" id="btnReset">♻️ Reset experiment</button>
                  </div>
                </div>

                <hr class="my-3">

                <div class="row g-2">
                  <div class="col-6">
                    <div class="p-2 bg-light rounded border">
                      <div class="small text-muted">Hidden true k (N/m):</div>
                      <div class="fw-bold" id="kTrue">— hidden —</div>
                      <div class="small text-muted">(Revealed after “Check my k”)</div>
                    </div>
                  </div>
                  <div class="col-6">
                    <div class="p-2 bg-light rounded border">
                      <div class="small text-muted">Your estimate k (N/m):</div>
                      <div class="fw-bold" id="kEst">—</div>
                      <div class="small" id="kFeedback">Add 3–6 readings for accuracy ⭐</div>
                    </div>
                  </div>
                </div>

              </div>
            </div>
          </div>

          <!-- Readings Table -->
          <div class="col-12">
            <div class="card shadow-sm">
              <div class="card-body">
                <h2 class="h6 mb-2">Readings Table 📋</h2>
                <div class="scroll-x">
                  <table class="table table-sm align-middle">
                    <thead class="table-light">
                      <tr>
                        <th>#</th>
                        <th>Mass (g)</th>
                        <th>Force F (N)</th>
                        <th>Extension x (m)</th>
                        <th>Note</th>
                        <th></th>
                      </tr>
                    </thead>
                    <tbody id="tblBody"></tbody>
                  </table>
                </div>
                <div class="d-flex justify-content-between">
                  <small class="text-muted">Tip: Spread masses 100–600 g for a good straight line.</small>
                  <small class="text-muted">Noise simulates real measurement ±1–3 mm.</small>
                </div>
              </div>
            </div>
          </div>

          <!-- F–x Plot -->
          <div class="col-12">
            <div class="card shadow-sm mt-3">
              <div class="card-body">
                <h2 class="h6 mb-2">Graph: Force F (N) vs Extension x (m) 📈</h2>
                <canvas id="plotFx" height="300" style="width:100%;"></canvas>
                <small class="text-muted">Best-fit line through origin: F = kx</small>
              </div>
            </div>
          </div>

          <!-- Theory & Tasks -->
          <div class="col-12">
            <div class="card shadow-sm">
              <div class="card-body">
                <h2 class="h6">Quick Theory 🧠</h2>
                <p class="mb-2">
                  Hooke’s Law: <strong>F = kx</strong>, where <em>F</em> is force (N), <em>x</em> is extension (m), and <em>k</em> is the spring constant (N/m).
                </p>
                <ol class="mb-2">
                  <li>Choose several masses (e.g., 100 g … 600 g).</li>
                  <li>Click <em>Measure</em> each time to log <em>F</em> and measured <em>x</em>.</li>
                  <li>Click <em>Compute k</em> to get your best-fit spring constant.</li>
                  <li>Click <em>Check my k</em> to see how close you are and earn ⭐.</li>
                </ol>
                <p class="mb-0"><strong>Elasticity demo:</strong> Watch the spring stretch visually as mass increases. Stay within elastic limit — very large mass may give a warning.</p>
              </div>
            </div>
          </div>

        </div><!-- /row -->
      </div><!-- /p -->
    </div><!-- /wrap -->
  </div>

  <script>
    // --- Setup ---
    const kTruePHP = <?= json_encode($k_true, JSON_UNESCAPED_UNICODE) ?>;
    const massEl = document.getElementById('mass');
    const springEl = document.getElementById('spring');
    const massInput = document.getElementById('massInput');
    const gInput = document.getElementById('gInput');
    const btnInc = document.getElementById('btnInc');
    const btnDec = document.getElementById('btnDec');
    const btnMeasure = document.getElementById('btnMeasure');
    const btnCompute = document.getElementById('btnCompute');
    const btnCheck = document.getElementById('btnCheck');
    const btnReset = document.getElementById('btnReset');
    const kTrueBox = document.getElementById('kTrue');
    const kEstBox = document.getElementById('kEst');
    const kFeedback = document.getElementById('kFeedback');
    const tblBody = document.getElementById('tblBody');

    const sceneBaseTop = 18;   // top of spring
    const baseSpringH  = 120;  // px base height (no load)
    const pxPerMeter   = 600;  // visual scaling
    let readings = [];

    // Ruler drawing (every 5 cm)
    const ruler = document.getElementById('ruler');
    for (let cm = 0; cm <= 50; cm += 5) {
      const y = 30 + (cm/100) * pxPerMeter;
      const tick = document.createElement('div');
      tick.className = 'tick';
      tick.style.top = `${y}px`;
      ruler.appendChild(tick);

      const lab = document.createElement('div');
      lab.className = 'label';
      lab.style.top = `${y}px`;
      lab.textContent = `${cm} cm`;
      ruler.appendChild(lab);
    }

    const clamp = (v, min, max) => Math.min(max, Math.max(min, v));
    function updateScene(){
      const m_g = clamp(parseInt(massInput.value || '0',10), 0, 1000);
      massInput.value = m_g;
      massEl.textContent = `${m_g} g`;

      const g = parseFloat(gInput.value || '9.81');
      const F = (m_g/1000) * g;        // N
      const x = (kTruePHP > 0) ? (F / kTruePHP) : 0; // m

      const springH = baseSpringH + x * pxPerMeter;
      springEl.style.height = `${springH}px`;
      massEl.style.top = `${sceneBaseTop + springH}px`;

      if (x > 0.25) { kFeedback.textContent = "Warning: close to elastic limit! Try smaller mass. ⚠️"; }
    }

    btnInc.addEventListener('click', () => { massInput.value = Math.min(1000, parseInt(massInput.value||'0',10) + 50); updateScene(); });
    btnDec.addEventListener('click', () => { massInput.value = Math.max(0, parseInt(massInput.value||'0',10) - 50); updateScene(); });
    massInput.addEventListener('input', updateScene);
    gInput.addEventListener('input', updateScene);

    btnMeasure.addEventListener('click', () => {
      const m_g = clamp(parseInt(massInput.value || '0',10), 0, 1000);
      const g   = parseFloat(gInput.value || '9.81');
      const F   = (m_g/1000) * g;             // Newtons
      const xId = F / kTruePHP;               // ideal extension (m)
      const noise = (Math.random()*0.003) * (Math.random()<0.5 ? -1 : 1); // ±3 mm
      const x    = Math.max(0, xId + noise);

      const note = (xId > 0.25) ? "⚠️ Near elastic limit" : "OK";
      readings.push({m_g, F, x, note});
      renderTable();
      kFeedback.textContent = "Nice! Add 3–6 readings then click Compute k. ⭐";
    });

    function renderTable(){
      tblBody.innerHTML = readings.map((r, i) => `
        <tr>
          <td>${i+1}</td>
          <td>${r.m_g}</td>
          <td>${r.F.toFixed(3)}</td>
          <td>${r.x.toFixed(3)}</td>
          <td>${r.note}</td>
          <td><button class="btn btn-sm btn-outline-danger" onclick="delRow(${i})">Delete</button></td>
        </tr>
      `).join('');
      drawFx(); // update chart on any table change
    }
    window.delRow = function(i){ readings.splice(i,1); renderTable(); }

    btnCompute.addEventListener('click', () => {
      if (readings.length < 2) { kFeedback.textContent = "Add more readings to compute k. 📈"; return; }
      let sumFx = 0, sumx2 = 0;
      readings.forEach(r => { sumFx += r.F * r.x; sumx2 += r.x * r.x; });
      if (sumx2 === 0){ kFeedback.textContent = "Extensions are zero — increase mass. 🤔"; return; }
      const kEst = sumFx / sumx2;
      kEstBox.textContent = kEst.toFixed(3);
      if (Math.abs(kEst - kTruePHP) / kTruePHP < 0.05)      kFeedback.textContent = "Excellent! Within 5% 🎉⭐";
      else if (Math.abs(kEst - kTruePHP) / kTruePHP < 0.10) kFeedback.textContent = "Good! Within 10% 🙌";
      else                                                  kFeedback.textContent = "Try more evenly spaced masses to improve accuracy. 🔁";
      drawFx();
    });

    btnCheck.addEventListener('click', () => {
      kTrueBox.textContent = kTruePHP.toFixed(3);
      const est = parseFloat(kEstBox.textContent);
      if (!isNaN(est)) {
        const err = Math.abs(est - kTruePHP)/kTruePHP;
        if (err < 0.05)      alert("🎉 Perfect lab! You earned ⭐⭐⭐");
        else if (err < 0.10) alert("👏 Nice work! You earned ⭐⭐");
        else                 alert("👍 Keep practicing! You earned ⭐");
      } else {
        alert("Compute k first to see your stars. 🌟");
      }
    });

    btnReset.addEventListener('click', () => {
      readings = [];
      renderTable();
      kEstBox.textContent = "—";
      kFeedback.textContent = "Add 3–6 readings for accuracy ⭐";
      window.location.reload();
    });

    // ---- F–x chart ----
    function drawFx(){
      const cvs = document.getElementById('plotFx');
      if (!cvs) return;
      const ctx = cvs.getContext('2d');
      const W = cvs.width = cvs.clientWidth;
      const H = cvs.height;

      const padL = 48, padB = 28, padR = 12, padT = 10;
      ctx.clearRect(0,0,W,H);

      const xs = readings.map(r=>r.x);
      const ys = readings.map(r=>r.F);
      const maxX = Math.max(0.05, ...(xs.length?xs:[0.05]));
      const maxY = Math.max(1, ...(ys.length?ys:[1]));

      function X(v){ return padL + (v/maxX) * (W-padL-padR); }
      function Y(v){ return H-padB - (v/maxY) * (H-padT-padB); }

      // axes
      ctx.strokeStyle = '#94a3b8'; ctx.lineWidth = 1;
      ctx.beginPath(); ctx.moveTo(padL, H-padB); ctx.lineTo(W-padR, H-padB); ctx.stroke();
      ctx.beginPath(); ctx.moveTo(padL, H-padB); ctx.lineTo(padL, padT); ctx.stroke();

      // ticks
      for(let i=1;i<=5;i++){
        const xx = X(i*maxX/5);
        ctx.beginPath(); ctx.moveTo(xx, H-padB); ctx.lineTo(xx, H-padB+6); ctx.stroke();
        ctx.fillStyle='#475569'; ctx.font='11px system-ui'; ctx.textAlign='center';
        ctx.fillText((i*maxX/5).toFixed(2), xx, H-padB+18);
      }
      for(let i=1;i<=5;i++){
        const yy = Y(i*maxY/5);
        ctx.beginPath(); ctx.moveTo(padL-6, yy); ctx.lineTo(padL, yy); ctx.stroke();
        ctx.fillStyle='#475569'; ctx.font='11px system-ui'; ctx.textAlign='right';
        ctx.fillText((i*maxY/5).toFixed(2), padL-8, yy+3);
      }
      ctx.fillStyle='#0f172a'; ctx.font='12px system-ui'; ctx.textAlign='center';
      ctx.fillText('Extension x (m)', (padL+W-padR)/2, H-6);
      ctx.save(); ctx.translate(14,(padT+H-padB)/2); ctx.rotate(-Math.PI/2);
      ctx.fillText('Force F (N)', 0, 0); ctx.restore();

      // points
      ctx.fillStyle='#0d6efd';
      readings.forEach(r=>{ ctx.beginPath(); ctx.arc(X(r.x), Y(r.F), 3, 0, Math.PI*2); ctx.fill(); });

      // best-fit line through origin: F = kx
      let sumFx=0,sumx2=0;
      readings.forEach(r=>{ sumFx+=r.F*r.x; sumx2+=r.x*r.x; });
      const k = (sumx2>0)? (sumFx/sumx2) : 0;
      ctx.strokeStyle = '#0ea5e9'; ctx.lineWidth = 2;
      ctx.beginPath(); ctx.moveTo(X(0), Y(0)); ctx.lineTo(X(maxX), Y(k*maxX)); ctx.stroke();
    }

    // initial
    updateScene();
    drawFx();
  </script>
</body>
</html>
