<?php
// ohms_law.php — Ohm’s Law Lab 🎮🔌 + Realistic Bulb 💡 + Analog Ammeter 🧭
// Circuit: Battery → Switch → Ammeter → Bulb (unknown R) → Battery
// Students vary V, see bulb glow (P=VI), analog needle fluctuates, fit I–V to estimate R.

session_start();
if (!isset($_SESSION['R_true'])) {
  // true resistance between 5 Ω and 50 Ω
  $_SESSION['R_true'] = mt_rand(50, 500) / 10.0; // 5.0–50.0 Ω
}
$R_true = (float)$_SESSION['R_true'];
?>
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <title>Ohm’s Law — Find R 🎮🔌</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">

  <style>
    body{ background:#eef7ff; }
    .wrap{ background:#fff; border-radius:1rem; box-shadow:0 6px 24px rgba(0,0,0,.06); overflow:hidden; }
    .top{ background:linear-gradient(135deg,#0d6efd 0%, #6ea8fe 100%); color:#fff; }
    .bread{ font-size:.9rem; }
    .panel{ background:#f8fbff; border:1px dashed #c6d8ff; border-radius:.75rem; padding:1rem; }
    .meter{ font-variant-numeric: tabular-nums; }
    .scroll-x{ overflow-x:auto; }
    table th, table td{ white-space:nowrap; }

    /* --- Circuit visuals --- */
    .circuit{
      display:flex; align-items:center; gap:14px; padding:12px;
      background:#fff; border-radius:.75rem; border:1px solid #e2e8f0;
    }
    .battery{
      width:60px; height:34px; background:linear-gradient(#4b5563,#111827);
      border-radius:6px; position:relative; box-shadow: inset 0 0 6px rgba(255,255,255,.08);
    }
    .battery::after{
      content:''; position:absolute; right:-6px; top:8px; width:6px; height:18px; background:#1f2937; border-radius:0 3px 3px 0;
    }
    .lead{ height:4px; background:#94a3b8; flex:1; border-radius:2px; position:relative; }

    /* Switch */
    .switch{
      width:70px; height:34px; background:#e2e8f0; border-radius:18px; position:relative; cursor:pointer;
      box-shadow: inset 0 2px 8px rgba(0,0,0,.08);
    }
    .knob{
      position:absolute; top:3px; left:3px; width:28px; height:28px; background:#fff; border-radius:50%;
      box-shadow:0 2px 6px rgba(0,0,0,.15); transition:left .2s ease, background .2s ease;
    }
    .switch.on .knob{ left:39px; background:#d1fae5; }
    .switch.on{ background:#86efac; }
    .swlabel{ position:absolute; top:50%; transform:translateY(-50%); right:8px; font-size:.7rem; color:#065f46; font-weight:600; }
    .swlabel.off{ right:auto; left:8px; color:#334155; }

    /* Analog ammeter frame */
    .ammeterBox{
      background:#fff; border:1px solid #e2e8f0; border-radius:10px; padding:8px 10px;
      width:170px;
      box-shadow:0 2px 8px rgba(2,6,23,.04);
      text-align:center;
    }
    .ammeterBox .title{ font-size:.75rem; color:#475569; margin-top:4px; }
    .ammeterBox .digital{ font-size:.85rem; color:#0f172a; margin-top:2px; }

    /* Bulb */
    .bulb{
      width:56px; height:56px; border-radius:50%; background:#f8fafc; border:2px solid #cbd5e1; position:relative;
      box-shadow: inset 0 0 12px rgba(0,0,0,.12);
    }
    .bulb.on{
      background: radial-gradient(closest-side, #fffbe6 0%, #ffe38f 45%, #ffd166 70%, #f8fafc 75%);
      border-color:#ffbf00;
      box-shadow:
        0 0 10px rgba(255,219,88,.6),
        0 0 28px rgba(255,182,66,.45),
        inset 0 0 12px rgba(255,255,255,.6);
    }
    .bulb .base{
      position:absolute; bottom:-8px; left:50%; transform:translateX(-50%);
      width:20px; height:10px; background:#64748b; border-radius:0 0 6px 6px;
    }

    /* Fuse badge */
    .fuse{ padding:4px 8px; border-radius:999px; font-size:.75rem; font-weight:600; }
    .fuse.ok{ background:#dcfce7; color:#065f46; border:1px solid #86efac; }
    .fuse.tripped{ background:#fee2e2; color:#991b1b; border:1px solid #fecaca; }
  </style>
</head>
<body>
<div class="container py-4">
  <a href="index.php" class="btn btn-link mb-3">⬅️ Back to index</a>

  <div class="wrap">
    <div class="top p-3">
      <div class="d-flex flex-wrap justify-content-between align-items-center gap-2">
        <div>
          <h1 class="h4 mb-0">Ohm’s Law — Find R 🎮🔌</h1>
          <small>I = V / R. Vary supply voltage, measure current, fit slope to estimate R. Bulb glows with P = VI; ammeter needle fluctuates.</small>
        </div>
        <span class="badge text-bg-light">SS2 • Electricity ✅</span>
      </div>
    </div>

    <div class="p-3 p-lg-4">
      <div class="row g-4">
        <!-- Visual Circuit -->
        <div class="col-12 col-lg-7">
          <div class="panel">
            <div class="bread mb-2">Circuit: Battery → <b>Switch</b> → <b>Ammeter</b> → <b>Bulb (unknown R)</b> → Battery</div>

            <div class="circuit mb-2">
              <div class="battery" title="Battery"></div>
              <div class="lead"></div>

              <!-- Switch -->
              <div id="switch" class="switch off" role="button" aria-pressed="false" title="Power switch">
                <div class="knob"></div>
                <span class="swlabel off">OFF</span>
              </div>

              <div class="lead"></div>

              <!-- Analog Ammeter -->
              <div class="ammeterBox" title="Ammeter">
                <svg id="ammeterSVG" viewBox="0 0 200 120" width="150" height="90">
                  <!-- semicircle scale -->
                  <path d="M20,100 A80,80 0 0,1 180,100" fill="none" stroke="#94a3b8" stroke-width="6"/>
                  <!-- ticks (drawn via JS) -->
                  <g id="ticks"></g>
                  <!-- needle pivot -->
                  <circle cx="100" cy="100" r="5" fill="#111827"/>
                  <!-- needle -->
                  <g id="needle" transform="rotate(-90 100 100)">
                    <line x1="100" y1="100" x2="100" y2="24" stroke="#ef4444" stroke-width="3" />
                  </g>
                </svg>
                <div class="title">Ammeter (0–2.5 A)</div>
                <div class="digital"><span id="ammeterVal">0.000</span> A</div>
              </div>

              <div class="lead"></div>

              <!-- Bulb -->
              <div id="bulb" class="bulb" title="Bulb (acts as unknown resistor)">
                <div class="base"></div>
              </div>
            </div>

            <div class="row g-2 align-items-center">
              <div class="col-12 col-md-4">
                <div class="small text-muted">Supply Voltage</div>
                <div class="display-6 meter" id="liveV">0.0 V</div>
              </div>
              <div class="col-12 col-md-4">
                <div class="small text-muted">Measured Current</div>
                <div class="display-6 meter" id="liveI">0.000 A</div>
              </div>
              <div class="col-12 col-md-4 d-flex align-items-center justify-content-between">
                <div>
                  <div class="small text-muted">Power (P = VI)</div>
                  <div class="h4 meter" id="liveP">0.00 W</div>
                </div>
                <span id="fuseBadge" class="fuse ok">Fuse: OK</span>
              </div>
            </div>

            <div class="mt-2">
              <input type="range" class="form-range" id="Vslider" min="0" max="12" step="0.1" value="0">
            </div>

            <small class="text-muted d-block mt-2">Tiny measurement noise added (±0.5% + ±0.002 A). Switch ON to energize circuit. Fuse trips above ~12 W.</small>
          </div>
        </div>

        <!-- Controls -->
        <div class="col-12 col-lg-5">
          <div class="card shadow-sm">
            <div class="card-body">
              <div class="row g-2">
                <div class="col-6">
                  <label class="form-label mb-1">Voltage V (manual)</label>
                  <div class="input-group">
                    <button class="btn btn-outline-secondary" id="btnVdec">−0.5</button>
                    <input type="number" class="form-control" id="Vin" value="0.0" step="0.1" min="0" max="12">
                    <button class="btn btn-outline-secondary" id="btnVinc">+0.5</button>
                  </div>
                  <div class="form-text">Use slider or buttons.</div>
                </div>

                <div class="col-6">
                  <label class="form-label mb-1">Auto-step sequence</label>
                  <div class="d-grid">
                    <button class="btn btn-outline-primary" id="btnSweep">🔁 Sweep 0→12 V (step 1V)</button>
                  </div>
                  <div class="form-text">Switch must be ON. Fuse may trip at high power.</div>
                </div>

                <div class="col-12 d-grid gap-2 mt-2">
                  <button class="btn btn-primary" id="btnMeasure">📸 Measure (add V–I)</button>
                  <button class="btn btn-success" id="btnCompute">📐 Compute R from fit</button>
                  <button class="btn btn-danger" id="btnReset">♻️ Reset</button>
                </div>
              </div>

              <hr class="my-3">
              <div class="row g-2">
                <div class="col-6">
                  <div class="p-2 bg-light rounded border">
                    <div class="small text-muted">Hidden true R (Ω):</div>
                    <div id="Rtrue">— hidden —</div>
                    <div class="small text-muted">(Revealed after compute)</div>
                  </div>
                </div>
                <div class="col-6">
                  <div class="p-2 bg-light rounded border">
                    <div class="small text-muted">Your estimate R (Ω):</div>
                    <div id="Rest">—</div>
                    <div class="small" id="feedback">Switch ON and collect 5–8 points ⭐</div>
                  </div>
                </div>
              </div>

            </div>
          </div>
        </div>

        <!-- Table -->
        <div class="col-12">
          <div class="card shadow-sm">
            <div class="card-body">
              <h2 class="h6 mb-2">V–I Readings 📋</h2>
              <div class="scroll-x">
                <table class="table table-sm">
                  <thead class="table-light">
                    <tr><th>#</th><th>V (V)</th><th>I (A)</th><th>P (W)</th><th></th></tr>
                  </thead>
                  <tbody id="tbl"></tbody>
                </table>
              </div>
            </div>
          </div>
        </div>

        <!-- I–V Plot -->
        <div class="col-12">
          <div class="card shadow-sm mt-3">
            <div class="card-body">
              <h2 class="h6 mb-2">Graph: Current I (A) vs Voltage V (V) 📈</h2>
              <canvas id="plotIV" height="300" style="width:100%;"></canvas>
              <small class="text-muted">Best-fit line through origin: I = (1/R)·V</small>
            </div>
          </div>
        </div>

        <!-- Theory -->
        <div class="col-12">
          <div class="card shadow-sm">
            <div class="card-body">
              <h2 class="h6">Quick Theory 🧠</h2>
              <p>Ohm’s law: <strong>V = IR</strong>. Plot <em>I</em> vs <em>V</em>, slope = <em>1/R</em>. So <strong>R = 1/slope</strong> for a line through the origin.</p>
            </div>
          </div>
        </div>

      </div>
    </div>
  </div>
</div>

<script>
const RtruePHP = <?= json_encode($R_true) ?>;

// UI refs
const Vslider = document.getElementById('Vslider');
const Vin     = document.getElementById('Vin');
const btnVinc = document.getElementById('btnVinc');
const btnVdec = document.getElementById('btnVdec');
const btnMeasure = document.getElementById('btnMeasure');
const btnSweep   = document.getElementById('btnSweep');
const btnCompute = document.getElementById('btnCompute');
const btnReset   = document.getElementById('btnReset');

const liveV = document.getElementById('liveV');
const liveI = document.getElementById('liveI');
const liveP = document.getElementById('liveP');
const ammeterVal = document.getElementById('ammeterVal');
const tbl   = document.getElementById('tbl');
const RtrueBox = document.getElementById('Rtrue');
const RestBox  = document.getElementById('Rest');
const feedback = document.getElementById('feedback');

const bulb = document.getElementById('bulb');
const sw = document.getElementById('switch');
const fuseBadge = document.getElementById('fuseBadge');

// Analog ammeter parts
const needle = document.getElementById('needle');
const ticksG = document.getElementById('ticks');
const I_MAX = 2.5;          // Ammeter full scale (A)
const ANG_MIN = -90;        // needle angle at 0 A
const ANG_MAX = 90;         // needle angle at I_MAX

// Draw tick marks (0 to 2.5 A, every 0.5 A major)
(function drawTicks() {
  for (let i=0; i<=10; i++) {
    const frac = i/10;                 // 0..1
    const Ival = frac * I_MAX;         // mapped to 0..2.5
    const ang = ANG_MIN + frac*(ANG_MAX-ANG_MIN);
    const rad = (Math.PI/180) * ang;
    const rOuter = 86, rInner = (i%2===0)? 74 : 80; // longer for major ticks
    const cx=100, cy=100;
    const x1 = cx + rInner*Math.cos(rad);
    const y1 = cy + rInner*Math.sin(rad);
    const x2 = cx + rOuter*Math.cos(rad);
    const y2 = cy + rOuter*Math.sin(rad);

    const line = document.createElementNS("http://www.w3.org/2000/svg","line");
    line.setAttribute("x1", x1); line.setAttribute("y1", y1);
    line.setAttribute("x2", x2); line.setAttribute("y2", y2);
    line.setAttribute("stroke", "#0f172a");
    line.setAttribute("stroke-width", (i%2===0)? "2" : "1");
    ticksG.appendChild(line);

    if (i%2===0){ // label each 0.5 A
      const rx = cx + 62*Math.cos(rad);
      const ry = cy + 62*Math.sin(rad) + 4;
      const txt = document.createElementNS("http://www.w3.org/2000/svg","text");
      txt.setAttribute("x", rx); txt.setAttribute("y", ry);
      txt.setAttribute("fill", "#0f172a");
      txt.setAttribute("font-size", "10");
      txt.setAttribute("text-anchor", "middle");
      txt.textContent = Ival.toFixed(1);
      ticksG.appendChild(txt);
    }
  }
})();

let points = [];        // {V,I}
let circuitOn = false;  // switch state
let fuseTripped = false;
const FUSE_WATT_LIMIT = 12.0; // trip at 12 W

// analog needle smoothing
let I_target = 0;         // current target (A) from meter
let I_display = 0;        // smoothed display (A)
let lastTime = performance.now();

// --- Switch behavior
sw.addEventListener('click', ()=>{
  if (fuseTripped) { alert("Fuse is tripped. Click Reset to replace fuse."); return; }
  circuitOn = !circuitOn;
  sw.classList.toggle('on', circuitOn);
  sw.setAttribute('aria-pressed', circuitOn ? 'true' : 'false');
  const old = sw.querySelector('.swlabel'); if (old) old.remove();
  const lbl = document.createElement('span'); lbl.className = 'swlabel ' + (circuitOn ? '' : 'off');
  lbl.textContent = circuitOn ? 'ON' : 'OFF';
  sw.appendChild(lbl);
  updateMeters(parseFloat(Vin.value||0));
});

// --- Meters & physics
function updateMeters(V){
  V = Math.max(0, Math.min(12, V));
  let I = 0;

  if (circuitOn && !fuseTripped) {
    const idealI = (RtruePHP > 0) ? (V / RtruePHP) : 0;
    const noisyI = Math.max(0, idealI*(1 + (Math.random()-0.5)*0.01) + (Math.random()-0.5)*0.004); // ±0.5% + ±0.002A
    I = noisyI;
  }

  const P = V * I;

  // live digital displays
  liveV.textContent = V.toFixed(1) + " V";
  liveI.textContent = I.toFixed(3) + " A";
  ammeterVal.textContent = I.toFixed(3);
  liveP.textContent = P.toFixed(2) + " W";

  // set analog target
  I_target = Math.min(I, I_MAX); // clamp to meter range

  // bulb glow
  const glow = Math.min(1, P / 10.0);
  if (I > 0 && !fuseTripped) {
    bulb.classList.add('on');
    bulb.style.boxShadow =
      `0 0 ${10 + glow*20}px rgba(255,219,88,${0.5+0.3*glow}),
       0 0 ${20 + glow*30}px rgba(255,182,66,${0.35+0.25*glow}),
       inset 0 0 12px rgba(255,255,255,.6)`;
  } else {
    bulb.classList.remove('on');
    bulb.style.boxShadow = 'inset 0 0 12px rgba(0,0,0,.12)';
  }

  // Fuse logic
  if (!fuseTripped && P > FUSE_WATT_LIMIT + (Math.random()*0.5)) {
    fuseTripped = true;
    circuitOn = false;
    sw.classList.remove('on');
    const old = sw.querySelector('.swlabel'); if (old) old.remove();
    const lbl = document.createElement('span'); lbl.className = 'swlabel off'; lbl.textContent = 'OFF'; sw.appendChild(lbl);

    fuseBadge.className = 'fuse tripped';
    fuseBadge.textContent = 'Fuse: TRIPPED';
    // drop meters
    liveI.textContent = '0.000 A';
    ammeterVal.textContent = '0.000';
    liveP.textContent = '0.00 W';
    I_target = 0;
    bulb.classList.remove('on');
    bulb.style.boxShadow = 'inset 0 0 12px rgba(0,0,0,.12)';
    feedback.textContent = "Fuse tripped (P too high). Reduce V and click Reset.";
  }

  return {V, I, P};
}

// Map current to needle angle
function currentToAngle(I){
  const frac = Math.max(0, Math.min(1, I / I_MAX));
  return ANG_MIN + frac * (ANG_MAX - ANG_MIN);
}

// Needle animation loop (damped, with small jitter)
function animateNeedle(ts){
  const dt = Math.min(0.05, (ts - lastTime)/1000); // cap dt for stability
  lastTime = ts;

  // critically-damped-ish smoothing towards target
  const speed = 6.0; // responsiveness
  I_display += (I_target - I_display) * (1 - Math.exp(-speed * dt));

  // jitter amplitude ~ small fraction of target (more when higher current)
  const jitterAmp = 0.01 + 0.01 * (I_target / I_MAX); // up to ~0.02 A
  const jitter = (Math.random()-0.5) * 2 * jitterAmp;
  const I_vis = Math.max(0, Math.min(I_MAX, I_display + jitter));

  const ang = currentToAngle(I_vis);
  needle.setAttribute('transform', `rotate(${ang} 100 100)`);

  requestAnimationFrame(animateNeedle);
}
requestAnimationFrame(animateNeedle);

// Slider/Input sync
function syncFromSlider(){
  Vin.value = parseFloat(Vslider.value).toFixed(1);
  updateMeters(parseFloat(Vslider.value));
}
function syncFromInput(){
  let v = parseFloat(Vin.value||0);
  if (isNaN(v)) v = 0;
  v = Math.max(0, Math.min(12, v));
  Vslider.value = v.toFixed(1);
  updateMeters(v);
}

Vslider.addEventListener('input', syncFromSlider);
Vin.addEventListener('input', syncFromInput);
btnVinc.addEventListener('click', ()=>{ Vin.value=(Math.min(12, parseFloat(Vin.value||0)+0.5)).toFixed(1); syncFromInput(); });
btnVdec.addEventListener('click', ()=>{ Vin.value=(Math.max(0, parseFloat(Vin.value||0)-0.5)).toFixed(1); syncFromInput(); });

// measure current at current V
btnMeasure.addEventListener('click', ()=>{
  if (!circuitOn) { alert("Switch is OFF. Turn it ON to measure."); return; }
  if (fuseTripped) { alert("Fuse is tripped. Click Reset to replace fuse."); return; }
  const V = parseFloat(Vin.value||0);
  const {I} = updateMeters(V);
  points.push({V, I});
  renderTable();
  feedback.textContent = "Great! Add more points and compute. ⭐";
});

btnSweep.addEventListener('click', async ()=>{
  if (!circuitOn) { alert("Switch is OFF. Turn it ON to sweep."); return; }
  points = [];
  for (let v=0; v<=12; v+=1){
    if (fuseTripped) break;
    Vin.value = v.toFixed(1);
    syncFromInput();
    const {I} = updateMeters(v);
    points.push({V:v, I});
    renderTable();
    await new Promise(r=>setTimeout(r, 120));
  }
  feedback.textContent = fuseTripped ? "Sweep stopped: Fuse tripped. Reset to continue." : "Sweep complete. Now compute R. 📐";
});

function renderTable(){
  tbl.innerHTML = points.map((p,i)=>`
    <tr>
      <td>${i+1}</td><td>${p.V.toFixed(1)}</td><td>${p.I.toFixed(3)}</td><td>${(p.V*p.I).toFixed(2)}</td>
      <td><button class="btn btn-sm btn-outline-danger" onclick="delRow(${i})">Delete</button></td>
    </tr>
  `).join('');
  drawIV(); // update chart whenever table changes
}
window.delRow = i => { points.splice(i,1); renderTable(); };

btnCompute.addEventListener('click', ()=>{
  // Fit I = m*V through origin => m = Σ(VI)/Σ(V^2); R = 1/m
  if (points.length < 3){
    feedback.textContent = "Take at least 3–5 points for a good line. 📈";
    return;
  }
  let sumVI = 0, sumV2 = 0;
  points.forEach(p=>{ sumVI += p.V*p.I; sumV2 += p.V*p.V; });
  if (sumV2 === 0){
    feedback.textContent = "All voltages are zero — increase V. ⚡";
    return;
  }
  const m = sumVI / sumV2;     // 1/R
  const Rest = (m>0) ? (1/m) : Infinity;
  RestBox.textContent = isFinite(Rest) ? Rest.toFixed(2) : "—";
  RtrueBox.textContent = RtruePHP.toFixed(2);

  const err = Math.abs(Rest - RtruePHP)/RtruePHP;
  if (err < 0.03)      { feedback.textContent = "🎉 Excellent! <3% error — ⭐⭐⭐"; alert("🎉 You earned ⭐⭐⭐"); }
  else if (err < 0.08) { feedback.textContent = "👏 Good! <8% error — ⭐⭐";       alert("👏 You earned ⭐⭐"); }
  else                 { feedback.textContent = "👍 Keep practicing — ⭐";          alert("👍 You earned ⭐"); }

  drawIV();
});

btnReset.addEventListener('click', ()=>{
  points = [];
  renderTable();
  RestBox.textContent = "—";
  feedback.textContent = "Collect 5–8 points for a good fit ⭐";
  // reset fuse & switch
  fuseTripped = false;
  fuseBadge.className = 'fuse ok';
  fuseBadge.textContent = 'Fuse: OK';
  circuitOn = false;
  sw.classList.remove('on');
  const old = sw.querySelector('.swlabel'); if (old) old.remove();
  const lbl = document.createElement('span'); lbl.className = 'swlabel off'; lbl.textContent = 'OFF'; sw.appendChild(lbl);
  bulb.classList.remove('on'); bulb.style.boxShadow = 'inset 0 0 12px rgba(0,0,0,.12)';
  // new R_true for variety
  location.reload();
});

// ------- I–V PLOT (Ohm) --------
function drawIV(){
  const cvs = document.getElementById('plotIV');
  if (!cvs) return;
  const ctx = cvs.getContext('2d');

  const W = cvs.width = cvs.clientWidth || 600;
  const H = cvs.height;

  const padL=48, padB=28, padR=12, padT=10;
  ctx.clearRect(0,0,W,H);

  function X(v, maxX){ return padL + (v/maxX)*(W-padL-padR); }
  function Y(v, maxY){ return H-padB - (v/maxY)*(H-padT-padB); }
  function axis(maxX, maxY){
    ctx.strokeStyle='#94a3b8'; ctx.lineWidth=1;
    ctx.beginPath(); ctx.moveTo(padL,H-padB); ctx.lineTo(W-padR,H-padB); ctx.stroke();
    ctx.beginPath(); ctx.moveTo(padL,H-padB); ctx.lineTo(padL,padT); ctx.stroke();
    const xt=5, yt=5;
    for(let i=1;i<=xt;i++){
      const xx=X(i*maxX/xt, maxX);
      ctx.beginPath(); ctx.moveTo(xx,H-padB); ctx.lineTo(xx,H-padB+6); ctx.stroke();
      ctx.fillStyle='#475569'; ctx.font='11px system-ui'; ctx.textAlign='center';
      ctx.fillText((i*maxX/xt).toFixed(1), xx, H-padB+18);
    }
    for(let i=1;i<=yt;i++){
      const yy=Y(i*maxY/yt, maxY);
      ctx.beginPath(); ctx.moveTo(padL-6,yy); ctx.lineTo(padL,yy); ctx.stroke();
      ctx.fillStyle='#475569'; ctx.font='11px system-ui'; ctx.textAlign='right';
      ctx.fillText((i*maxY/yt).toFixed(3), padL-8, yy+3);
    }
    ctx.fillStyle='#0f172a'; ctx.font='12px system-ui';
    ctx.textAlign='center'; ctx.fillText('Voltage V (V)', (padL+W-padR)/2, H-6);
    ctx.save(); ctx.translate(14,(padT+H-padB)/2); ctx.rotate(-Math.PI/2);
    ctx.fillText('Current I (A)', 0, 0); ctx.restore();
  }

  if (points.length === 0){ axis(12, 0.5); return; }

  const xs = points.map(p=>p.V);
  const ys = points.map(p=>p.I);
  const maxX = Math.max(...xs, 1);
  const maxY = Math.max(...ys, 0.1);

  let sumVI=0, sumV2=0;
  points.forEach(p=>{ sumVI+=p.V*p.I; sumV2+=p.V*p.V; });
  const m = (sumV2>0)? (sumVI/sumV2) : 0;

  axis(maxX, maxY);

  ctx.fillStyle='#0d6efd';
  points.forEach(p=>{
    ctx.beginPath(); ctx.arc(X(p.V,maxX), Y(p.I,maxY), 3, 0, Math.PI*2); ctx.fill();
  });

  ctx.strokeStyle='#0ea5e9'; ctx.lineWidth=2;
  ctx.beginPath();
  ctx.moveTo(X(0,maxX), Y(0,maxY));
  ctx.lineTo(X(maxX,maxX), Y(m*maxX,maxY));
  ctx.stroke();
}

// init
const lbl = document.createElement('span'); lbl.className='swlabel off'; lbl.textContent='OFF'; sw.appendChild(lbl);
syncFromSlider();
drawIV();
window.addEventListener('resize', drawIV);
</script>
</body>
</html>
