<?php
// simple_pendulum.php — Simple Pendulum Lab 🎮⏱️
// Find g using T = 2π√(L/g). Students time N oscillations to get T, then estimate g.

session_start();
if (!isset($_SESSION['g_true'])) {
  // pick true g (m/s^2) around 9.81 ± 0.15 to simulate location/conditions
  $_SESSION['g_true'] = 9.81 + (mt_rand(-15, 15) / 100.0);
}
$g_true = (float)$_SESSION['g_true'];
?>
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <title>Simple Pendulum — Find g 🎮⏱️</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <style>
    body{ background:#eef7ff; }
    .wrap{ background:#fff; border-radius:1rem; box-shadow:0 6px 24px rgba(0,0,0,.06); overflow:hidden; }
    .top{ background:linear-gradient(135deg,#198754 0%, #6ed6a0 100%); color:#fff; }
    .scene{ position:relative; height:300px; background: #f8fbff; }
    .pivot{ position:absolute; top:8px; left:50%; transform:translateX(-50%); width:12px; height:12px; background:#198754; border-radius:50%; }
    .string{ position:absolute; top:14px; left:50%; width:2px; background:#0d6efd; transform-origin: top center; }
    .bob{ position:absolute; width:40px; height:40px; background:#2b3035; color:#fff; border-radius:50%; display:flex; align-items:center; justify-content:center; font-weight:700; transform:translate(-50%, -50%); }
    .scroll-x{ overflow-x:auto; }
    table th, table td{ white-space:nowrap; }
  </style>
</head>
<body>
<div class="container py-4">
  <a href="index.php" class="btn btn-link mb-3">⬅️ Back to index</a>

  <div class="wrap">
    <div class="top p-3">
      <div class="d-flex flex-wrap justify-content-between align-items-center gap-2">
        <div>
          <h1 class="h4 mb-0">Simple Pendulum — Find g 🎮⏱️</h1>
          <small>Use time for N oscillations: g = (4π² L) / T²</small>
        </div>
        <span class="badge text-bg-light">SS2 • Mechanics ✅</span>
      </div>
    </div>

    <div class="p-3 p-lg-4">
      <div class="row g-4">
        <!-- Visual -->
        <div class="col-12 col-lg-6">
          <div class="scene rounded border">
            <div class="pivot"></div>
            <div id="string" class="string" style="height:200px; transform:rotate(0deg) translateX(-1px);"></div>
            <div id="bob" class="bob" style="top:214px; left:50%;">m</div>
          </div>
          <div class="text-center text-muted mt-2">
            <small>Tap <b>Start</b>, wait N swings, then <b>Stop</b>. Or use Auto-measure 🤖.</small>
          </div>
        </div>

        <!-- Controls -->
        <div class="col-12 col-lg-6">
          <div class="card shadow-sm">
            <div class="card-body">
              <div class="row g-2">
                <div class="col-6">
                  <label class="form-label mb-1">Length L (m) 🎣</label>
                  <input type="number" class="form-control" id="L" value="1.000" step="0.001" min="0.20" max="1.50">
                  <div class="form-text">0.20–1.50 m recommended.</div>
                </div>
                <div class="col-6">
                  <label class="form-label mb-1">Oscillations N 🔁</label>
                  <input type="number" class="form-control" id="N" value="20" step="1" min="5" max="50">
                  <div class="form-text">Time several swings to average out error.</div>
                </div>

                <div class="col-12 d-grid gap-2 mt-2">
                  <button class="btn btn-primary" id="btnStart">▶️ Start Timer</button>
                  <button class="btn btn-warning" id="btnStop" disabled>⏹️ Stop & Record</button>
                  <button class="btn btn-outline-primary" id="btnAuto">🤖 Auto-measure (simulate)</button>
                  <button class="btn btn-success" id="btnCompute">📐 Compute g from readings</button>
                  <button class="btn btn-danger" id="btnReset">♻️ Reset</button>
                </div>
              </div>

              <hr class="my-3">
              <div class="row g-2">
                <div class="col-6">
                  <div class="p-2 bg-light rounded border">
                    <div class="small text-muted">Hidden true g (m/s²):</div>
                    <div id="gTrue">— hidden —</div>
                    <div class="small text-muted">(Revealed after compute)</div>
                  </div>
                </div>
                <div class="col-6">
                  <div class="p-2 bg-light rounded border">
                    <div class="small text-muted">Your estimate g (m/s²):</div>
                    <div id="gEst">—</div>
                    <div class="small" id="feedback">Get 3+ readings for accuracy ⭐</div>
                  </div>
                </div>

                <!-- Graph-fit g box -->
                <div class="col-12 mt-2">
                  <div class="p-2 bg-light rounded border">
                    <div class="small text-muted">Graph-fit g (from T² vs L):</div>
                    <div id="gFit">—</div>
                    <div class="small text-muted">Slope s = T²/L ⇒ g = 4π² / s</div>
                  </div>
                </div>
              </div>

            </div>
          </div>
        </div>

        <!-- Readings -->
        <div class="col-12">
          <div class="card shadow-sm">
            <div class="card-body">
              <h2 class="h6 mb-2">Readings (time for N oscillations) 📋</h2>
              <div class="scroll-x">
                <table class="table table-sm">
                  <thead class="table-light">
                    <tr>
                      <th>#</th><th>L (m)</th><th>N</th><th>Time (s)</th><th>Period T (s)</th><th></th>
                    </tr>
                  </thead>
                  <tbody id="tbl"></tbody>
                </table>
              </div>
              <small class="text-muted">We add small timing noise (human reaction ±0.05s).</small>
            </div>
          </div>
        </div>

        <!-- T²–L Plot -->
        <div class="col-12">
          <div class="card shadow-sm mt-3">
            <div class="card-body">
              <h2 class="h6 mb-2">Graph: T² (s²) vs Length L (m) 📈</h2>
              <canvas id="plotT2L" height="300" style="width:100%;"></canvas>
              <small class="text-muted">For a simple pendulum: T = 2π√(L/g) ⇒ T² = (4π²/g)L</small>
            </div>
          </div>
        </div>

        <!-- Theory -->
        <div class="col-12">
          <div class="card shadow-sm">
            <div class="card-body">
              <h2 class="h6">Quick Theory 🧠</h2>
              <p>For small amplitudes, <strong>T = 2π√(L/g)</strong>. Measure time for N oscillations, <em>T = t/N</em>, then <strong>g = (4π² L)/T²</strong>.</p>
            </div>
          </div>
        </div>

      </div>
    </div>
  </div>
</div>

<script>
const gTruePHP = <?= json_encode($g_true) ?>;
const L = document.getElementById('L');
const N = document.getElementById('N');
const btnStart = document.getElementById('btnStart');
const btnStop  = document.getElementById('btnStop');
const btnAuto  = document.getElementById('btnAuto');
const btnCompute = document.getElementById('btnCompute');
const btnReset = document.getElementById('btnReset');
const gTrueBox = document.getElementById('gTrue');
const gEstBox  = document.getElementById('gEst');
const feedback = document.getElementById('feedback');
const tbl = document.getElementById('tbl');

const stringEl = document.getElementById('string');
const bobEl    = document.getElementById('bob');

let startTime = 0;
let readings = [];

// simple swing animation feedback (visual only)
let angle = 12, dir = 1;
setInterval(()=> {
  angle += dir*0.6;
  if (angle > 12 || angle < -12) dir *= -1;
  const Lpx = Math.max(120, Math.min(260, parseFloat(L.value||1)*200));
  stringEl.style.height = Lpx + 'px';
  stringEl.style.transform = `rotate(${angle}deg) translateX(-1px)`;
  const x = 50 + Math.sin(angle*Math.PI/180)*Lpx;
  const y = 14 + Math.cos(angle*Math.PI/180)*Lpx;
  bobEl.style.left = x + '%';
  bobEl.style.top  = (y + 14) + 'px';
}, 30);

// Timer buttons
btnStart.addEventListener('click', ()=>{
  startTime = performance.now();
  btnStart.disabled = true;
  btnStop.disabled = false;
  feedback.textContent = "Timing… swing, swing, swing… ⏱️";
});
btnStop.addEventListener('click', ()=>{
  if (!startTime) return;
  const end = performance.now();
  const t = (end - startTime)/1000;
  startTime = 0;
  btnStart.disabled = false;
  btnStop.disabled = true;

  const Lm = Math.max(0.2, Math.min(1.5, parseFloat(L.value || 1)));
  const n  = Math.max(5, Math.min(50, parseInt(N.value || 20,10)));
  const T  = t / n;
  readings.push({L: Lm, N: n, t, T});
  renderTable();
  feedback.textContent = "Nice! Add 2+ readings then Compute. ⭐";
});

// Auto simulate using g_true + tiny noise
btnAuto.addEventListener('click', ()=>{
  const Lm = Math.max(0.2, Math.min(1.5, parseFloat(L.value || 1)));
  const n  = Math.max(5, Math.min(50, parseInt(N.value || 20,10)));
  const Tideal = 2*Math.PI*Math.sqrt(Lm / gTruePHP);
  // human reaction ±0.05s per run, plus tiny model noise
  const t = n*Tideal + (Math.random()*0.10 - 0.05);
  readings.push({L: Lm, N: n, t, T: t/n});
  renderTable();
  feedback.textContent = "Simulated reading added. ⭐";
});

function renderTable(){
  tbl.innerHTML = readings.map((r,i)=>`
    <tr>
      <td>${i+1}</td>
      <td>${r.L.toFixed(3)}</td>
      <td>${r.N}</td>
      <td>${r.t.toFixed(2)}</td>
      <td>${r.T.toFixed(3)}</td>
      <td><button class="btn btn-sm btn-outline-danger" onclick="delRow(${i})">Delete</button></td>
    </tr>
  `).join('');
  drawT2L(); // update chart whenever table changes
}
window.delRow = (i)=>{ readings.splice(i,1); renderTable(); };

// Compute g: average across readings (simple mean)
btnCompute.addEventListener('click', ()=>{
  if (readings.length < 2){
    feedback.textContent = "Add at least 2–3 readings for stable g. 📈";
    return;
  }
  let gSum = 0;
  readings.forEach(r=>{
    const g = (4*Math.PI*Math.PI*r.L) / (r.T*r.T);
    gSum += g;
  });
  const gEst = gSum / readings.length;
  gEstBox.textContent = gEst.toFixed(3);
  gTrueBox.textContent = gTruePHP.toFixed(3);

  const err = Math.abs(gEst - gTruePHP)/gTruePHP;
  if (err < 0.02)      { feedback.textContent = "🎉 Superb! <2% error — ⭐⭐⭐"; alert("🎉 You earned ⭐⭐⭐"); }
  else if (err < 0.05) { feedback.textContent = "👏 Great! <5% error — ⭐⭐";  alert("👏 You earned ⭐⭐"); }
  else                 { feedback.textContent = "👍 Keep practicing — ⭐";     alert("👍 You earned ⭐"); }

  drawT2L(); // refresh chart (ensures line uses all readings)
});

btnReset.addEventListener('click', ()=>{
  readings = [];
  renderTable();
  gEstBox.textContent = "—";
  document.getElementById('gFit').textContent = "—";
  feedback.textContent = "Get 3+ readings for accuracy ⭐";
  location.reload(); // new g_true
});

// ------- T²–L PLOT (Pendulum) + graph-fit g --------
function drawT2L(){
  const cvs = document.getElementById('plotT2L');
  if (!cvs) return;
  const ctx = cvs.getContext('2d');
  const W = cvs.width = cvs.clientWidth || 600; // responsive width
  const H = cvs.height;                          // fixed via attribute (300)
  const padL=48, padB=28, padR=12, padT=10;
  ctx.clearRect(0,0,W,H);

  function X(v, maxX){ return padL + (v/maxX)*(W-padL-padR); }
  function Y(v, maxY){ return H-padB - (v/maxY)*(H-padT-padB); }
  function axis(maxX, maxY){
    ctx.strokeStyle='#94a3b8'; ctx.lineWidth=1;
    // x-axis
    ctx.beginPath(); ctx.moveTo(padL,H-padB); ctx.lineTo(W-padR,H-padB); ctx.stroke();
    // y-axis
    ctx.beginPath(); ctx.moveTo(padL,H-padB); ctx.lineTo(padL,padT); ctx.stroke();
    // ticks
    const xt=5, yt=5;
    for(let i=1;i<=xt;i++){
      const xx=X(i*maxX/xt, maxX);
      ctx.beginPath(); ctx.moveTo(xx,H-padB); ctx.lineTo(xx,H-padB+6); ctx.stroke();
      ctx.fillStyle='#475569'; ctx.font='11px system-ui'; ctx.textAlign='center';
      ctx.fillText((i*maxX/xt).toFixed(2), xx, H-padB+18);
    }
    for(let i=1;i<=yt;i++){
      const yy=Y(i*maxY/yt, maxY);
      ctx.beginPath(); ctx.moveTo(padL-6,yy); ctx.lineTo(padL,yy); ctx.stroke();
      ctx.fillStyle='#475569'; ctx.font='11px system-ui'; ctx.textAlign='right';
      ctx.fillText((i*maxY/yt).toFixed(2), padL-8, yy+3);
    }
    // labels
    ctx.fillStyle='#0f172a'; ctx.font='12px system-ui';
    ctx.textAlign='center'; ctx.fillText('Length L (m)', (padL+W-padR)/2, H-6);
    ctx.save(); ctx.translate(14,(padT+H-padB)/2); ctx.rotate(-Math.PI/2);
    ctx.fillText('T² (s²)', 0, 0); ctx.restore();
  }

  if (readings.length===0){
    axis(1,1); // empty axes
    return;
  }

  // Build (L, T²) points
  const pts = readings.map(r=>({ L: r.L, T2: r.T*r.T }));
  const maxX = Math.max(...pts.map(p=>p.L), 0.5);
  const maxY = Math.max(...pts.map(p=>p.T2), 2.0);

  // Fit line through origin: T² = s*L,  s = Σ(L*T²)/Σ(L²)
  let sumLT2=0, sumL2=0;
  pts.forEach(p=>{ sumLT2+=p.L*p.T2; sumL2+=p.L*p.L; });
  const s = (sumL2>0)? (sumLT2/sumL2) : 0;
  const gFit = (s>0)? (4*Math.PI*Math.PI/s) : NaN;
  const gFitBox = document.getElementById('gFit');
  if (gFitBox) gFitBox.textContent = (isFinite(gFit) ? gFit.toFixed(3) : '—');

  axis(maxX, maxY);

  // points
  ctx.fillStyle='#0d6efd';
  pts.forEach(p=>{
    ctx.beginPath(); ctx.arc(X(p.L,maxX), Y(p.T2,maxY), 3, 0, Math.PI*2); ctx.fill();
  });

  // best-fit line: T² = sL
  ctx.strokeStyle='#0ea5e9'; ctx.lineWidth=2;
  ctx.beginPath();
  ctx.moveTo(X(0,maxX), Y(0,maxY));
  ctx.lineTo(X(maxX,maxX), Y(s*maxX,maxY));
  ctx.stroke();
}

// init
drawT2L(); // initial empty axes
window.addEventListener('resize', drawT2L);
</script>
</body>
</html>
